<?php


namespace Olympus\Test\Http;


use Olympus\Contracts\CanMakeCall;
use Olympus\Helpers\RequestBag;
use Olympus\Http\RequestBuilder;
use Olympus\Test\BaseTest;
use Psr\Http\Message\RequestInterface;

class RequestBuilderTest extends BaseTest
{
    public function getRoute()
    {
        return [
            'method' => 'get',
            'path' => ['id'],
            'required' => [
                'path' => ['id'],
                'query' => ['start'],
                'params' => ['name', 'dob']
            ],
            'endpoint' => '/{id}/users',
            'query' => ['start', 'finish'],
            'auth' => true,
            'params' => ['age', 'dob', 'name']
        ];
    }

    public function getFullPayload()
    {
        $pathParams = [
            'id' => 1
        ];
        $queryParams = [
            'start' => 200,
            'finish' => 300
        ];
        $bodyParams = [
            'name' => 'test',
            'dob' => '2020-04-17',
            'age' => 2,
        ];

        return new RequestBag($pathParams, $queryParams, $bodyParams);
    }
    public function testExtractPathParams()
    {
        $actor = \Mockery::mock(CanMakeCall::class);

        $route = $this->getRoute();

        $payload = $this->getFullPayload();

        $builder = new RequestBuilder(
            $actor,
            $route,
            $payload
        );
        $reflection = new \ReflectionClass(RequestBuilder::class);
        $method = $reflection->getMethod('extractParams');
        $method->setAccessible(true);

        $pathParams = $method->invokeArgs($builder, ['path']);

        $this->assertEquals(['id' => 1], $pathParams);
    }

    public function testExtractQueryParams()
    {
        $actor = \Mockery::mock(CanMakeCall::class);

        $route = $this->getRoute();

        $payload = $this->getFullPayload();

        $builder = new RequestBuilder(
            $actor,
            $route,
            $payload
        );

        $reflection = new \ReflectionClass(RequestBuilder::class);
        $method = $reflection->getMethod('extractQueryParams');
        $method->setAccessible(true);

        $obj = $method->invoke($builder);

        $res = $obj->getQueryParams();
        $this->assertEquals([
            'start' => 200,
            'finish' => 300
        ], $res);
    }

    public function testExtractBodyParams()
    {
        $actor = \Mockery::mock(CanMakeCall::class);

        $route = $this->getRoute();

        $payload = $this->getFullPayload();

        $builder = new RequestBuilder(
            $actor,
            $route,
            $payload
        );

        $reflection = new \ReflectionClass(RequestBuilder::class);
        $method = $reflection->getMethod('extractBodyParams');
        $method->setAccessible(true);

        $obj = $method->invoke($builder);

        $res = $obj->getBodyParams();
        $this->assertEquals([
            'age' => 2,
            'dob' => '2020-04-17',
            'name' => 'test'
        ], $res);
    }

    public function testBuild()
    {
        $url = 'http://someurl.com';
        $authHeader = "Bearer token";
        $actor = \Mockery::mock(CanMakeCall::class);
        $actor
            ->shouldReceive('getAuthorizationHeader')
            ->andReturn($authHeader);
        $actor->shouldReceive('getApiUrl')
            ->andReturn($url);

        $route = $this->getRoute();

        $payload = $this->getFullPayload();

        $builder = new RequestBuilder(
            $actor,
            $route,
            $payload
        );

        $builder->build();

        $request = $builder->getRequest();

        $this->assertInstanceOf(RequestInterface::class, $request);

        $body = (string) $request->getBody();
        $body = json_decode($body, true);

        $this->assertEquals([
            'age' => 2,
            'dob' => '2020-04-17',
            'name' => 'test'
        ], $body);

        $expectedUrl = $url . '/1/users?start=200&finish=300';
        $path = (string) $request->getUri();

        $this->assertEquals($expectedUrl, $path);

        $this->assertEquals('get', strtolower($request->getMethod()));

        $authorizationHeader = $request->getHeader('Authorization')[0];
        $this->assertEquals($authHeader, $authorizationHeader);
    }
}
