<?php


namespace Olympus\Test\Actors;


use Olympus\Actors\User;
use Olympus\Api;
use Olympus\Test\BaseTest;

class UserTest extends BaseTest
{
    public function testAuthenticate()
    {
        $apiUrl = "http://somedummyapi.com";
        $clientId = "some fake id";
        $clientSecret = "some fake secret";
        $accessToken = "sometoken";
        $expiresIn = 1000000;
        $refreshToken = "somerefreshtoken";
        $uid = 1;
        $email = "hello@mobnia.com";

        $returnData = [
            "data" => [
                "token_data" => [
                    "access_token" => $accessToken,
                    "expires_in" => $expiresIn,
                    "refresh_token" => $refreshToken
                ],
                "user" => [
                    'id' => $uid,
                    'email' => $email
                ]
            ]
        ];

        $api = new Api($apiUrl, $clientId, $clientSecret);
        $caller = $this->getCallerMock($returnData);
        $caller->shouldReceive('getActor')
            ->andReturn($api);
        $api->setCaller($caller);

        $authType = "password_authentication";
        $authParams = [
            "uid" => $email,
            "password" => "fakepassword"
        ];

        $user = new User($api);
        $user->authenticate([
            'auth_type' => $authType,
            'auth_params'=> $authParams
        ]);

        $this->assertEquals($accessToken, $user->getAccessToken());
        $this->assertEquals($refreshToken, $user->getRefreshToken());
        $this->assertEquals($uid, $user->getUid());

        $diff = time() + $expiresIn - $user->getExpiresAt();
        $this->assertEquals(0, $diff);
    }

    public function testRefreshToken()
    {
        $apiUrl = "http://somedummyapi.com";
        $clientId = "some fake id";
        $clientSecret = "some fake secret";
        $accessToken = "sometoken";
        $expiresIn = 1000000;
        $refreshToken = "somerefreshtoken";
        $newRefreshToken = "new_refresh_token";
        $newAccessToken = "new_access_token";
        $newExpiresIn = 10000000;
        $uid = 1;

        $returnData = [
            "data" => [
                "access_token" => $newAccessToken,
                "expires_in" => $newExpiresIn,
                "refresh_token" => $newRefreshToken
            ]
        ];

        $api = new Api($apiUrl, $clientId, $clientSecret);
        $caller = $this->getCallerMock($returnData);
        $caller->shouldReceive('getActor')
            ->andReturn($api);
        $api->setCaller($caller);

        $user = \Mockery::mock(User::class, [
            $api,
            $accessToken,
            $expiresIn + time(),
            $uid,
            $refreshToken,
            "read write"
        ])
            ->shouldAllowMockingProtectedMethods()
            ->makePartial();

        $this->assertEquals($accessToken, $user->getAccessToken());

        $user->refreshAccessToken();

        $this->assertEquals($newAccessToken, $user->getAccessToken());
        $this->assertEquals($newRefreshToken, $user->getRefreshToken());
        $this->assertEquals(time() + $newExpiresIn, $user->getExpiresAt());
    }

    public function testCallsRefreshTokenIfAccessTokenExpired()
    {
        $accessToken = "sometoken";
        $expiresIn = 1000000;
        $refreshToken = "somerefreshtoken";
        $uid = 1;

        $api = \Mockery::mock(Api::class);

        $user = \Mockery::mock(User::class, [
            $api,
            $accessToken,
            time() - $expiresIn, // access token expired
            $uid,
            $refreshToken,
            "read write"
        ])
            ->shouldAllowMockingProtectedMethods()
            ->makePartial();

        $user->shouldReceive('refreshAccessToken');

        $user->getAccessToken();

        $user->shouldHaveReceived('refreshAccessToken');
    }
}
