<?php


namespace Olympus;

use GuzzleHttp\Exception\GuzzleException;
use Olympus\Actors\Client;
use Olympus\Actors\User;
use Olympus\Contracts\CanMakeCall;
use Olympus\Helpers\RequestBag;
use Olympus\Http\Caller;
use Olympus\Http\MagicCallerTrait;

class Api implements CanMakeCall
{
    use MagicCallerTrait;

    const PASSWORD_AUTH_TYPE = "password_authentication";

    /**
     * @var string
     */
    protected $apiUrl;

    /**
     * @var
     */
    protected $clientId;

    /**
     * @var
     */
    protected $clientSecret;

    /**
     * @var string
     */
    protected $logfile;

    /**
     * @var string
     */
    protected $logformat;

    /**
     * @var
     */
    protected $caller;

    /**
     * Api constructor.
     * @param $apiUrl
     * @param $clientId
     * @param $clientSecret
     * @param null $logfile
     * @param null $logformat
     */
    public function __construct($apiUrl, $clientId, $clientSecret, $logfile = null, $logformat = null)
    {
        $this->apiUrl = $apiUrl;
        $this->clientId = $clientId;
        $this->clientSecret = $clientSecret;
        $this->logfile = $logfile;
        $this->logformat = $logformat ?? Caller::DEFAULT_LOG_FORMAT;
    }

    /**
     * @return string
     */
    public function getApiUrl(): string
    {
        return $this->apiUrl;
    }

    /**
     * @return string
     */
    public function getClientId(): string
    {
        return $this->clientId;
    }

    /**
     * @return string
     */
    public function getClientSecret(): string
    {
        return $this->clientSecret;
    }

    /**
     * @param string $apiUrl
     * @return $this
     */
    public function setApiUrl(string $apiUrl): self
    {
        $this->apiUrl = $apiUrl;
        return  $this;
    }

    /**
     * @param string $clientId
     * @return $this
     */
    public function setClientId(string $clientId): self
    {
        $this->clientId = $clientId;
        return $this;
    }

    /**
     * @param string $clientSecret
     * @return $this
     */
    public function setClientSecret(string $clientSecret): self
    {
        $this->clientSecret = $clientSecret;
        return $this;
    }

    public function getLogFile()
    {
        return $this->logfile;
    }

    public function getLogFormat()
    {
        return $this->logformat;
    }

    public function setLogFile(string $logfile)
    {
        $this->logfile = $logfile;
        return $this;
    }

    public function setLogFormat(string $logformat)
    {
        $this->logformat = $logformat;
        return $this;
    }

    public function getActorForToken($accessToken)
    {
        $bag = new RequestBag([], [], ['access_token' => $accessToken]);
        $tokenData = $this->auth->whoAmI($bag);
        $tokenData = $tokenData->data;

        $scopes = $tokenData->token_data->scopes;
        $scopes = is_string($scopes) ? $scopes : implode(" ", $scopes);

        if ($tokenData->user) {
            $actor = new User(
                $this,
                $tokenData->token_data->access_token,
                (new \DateTime())->getTimestamp() + $tokenData->token_data->expires_in,
                $tokenData->user->id,
                $tokenData->token_data->refresh_token,
                $scopes
            );
            $actor->setUserAuthData($tokenData->user);
        } else {
            $actor = new Client(
                $this,
                $tokenData->token_data->access_token,
                (new \DateTime())->getTimestamp() + $tokenData->token_data->expires_in,
                null,
                $tokenData->token_data->refresh_token,
                $scopes
            );
        }

        return $actor;
    }

    /**
     * @return Client
     * @throws Exception\OlympusException
     * @throws GuzzleException
     */
    public function getClientActor()
    {
        $client =  new Client($this);
        $client->authenticate([
            "client_id" => $this->getClientId(),
            "client_secret" => $this->getClientSecret()
        ]);
        return $client;
    }

    /**
     * @param string $authType
     * @param array $authParams
     * @return User
     * @throws Exception\OlympusException
     * @throws GuzzleException
     */
    public function getUserActor(string $authType, array $authParams)
    {
        $user = new User($this);

        $user->authenticate([
            'auth_type' => $authType,
            'auth_params' => $authParams
        ]);

        return $user;
    }

    /**
     * @return string|null
     */
    public function getAuthorizationHeader(): ?string
    {
        $clientId = $this->getClientId();
        $clientSecret = $this->getClientSecret();
        if (!$clientId || !$clientSecret) {
            return null;
        }
        return 'Basic ' .base64_encode($clientId . ":". $clientSecret);
    }

    public function getCaller()
    {
        if (!$this->caller) {
            $this->caller = new Caller($this, $this->getLogFile(), $this->getLogFormat());
        }

        return $this->caller;
    }

    public function setCaller(Caller $caller)
    {
        $this->caller = $caller;
    }

    public function getUid(): ?string
    {
        return null;
    }
}
