<?php


namespace Olympus\Test\Http;


use GuzzleHttp\Client;
use GuzzleHttp\Promise\FulfilledPromise;
use Olympus\Contracts\CanMakeCall;
use Olympus\Helpers\RequestBag;
use Olympus\Http\Caller;
use Olympus\Test\BaseTest;
use Psr\Http\Message\ResponseInterface;

class CallerTest extends BaseTest
{
    public function testGetGuzzleClient()
    {
        $actor = \Mockery::mock(CanMakeCall::class);

        $caller = new Caller($actor, 'log');

        $reflection = new \ReflectionClass(Caller::class);
        $method = $reflection->getMethod('getGuzzleClient');
        $method->setAccessible(true);

        $client = $method->invoke($caller);

        $this->assertInstanceOf(Client::class, $client);
    }

    public function testCallEndpoint()
    {
        $return = [
            'id' => 1
        ];

        $authHeader = "Bearer sometoken";
        $url = "http://someurl.com";

        $actor = \Mockery::mock(CanMakeCall::class);
        $actor
            ->shouldReceive('getAuthorizationHeader')
            ->andReturn($authHeader);
        $actor->shouldReceive('getApiUrl')
            ->andReturn($url);

        $caller = \Mockery::mock(Caller::class, [$actor])
            ->makePartial()
            ->shouldAllowMockingProtectedMethods();

        $caller->shouldReceive('getGuzzleClient')
            ->andReturn($this->getGuzzleMock($return));

        $route = [
            'endpoint' => '/',
            'method' => 'get'
        ];
        $payload = new RequestBag();

        $res = $caller->callEndPoint($route, $payload);

        $this->assertEquals((object) $return, $res);
    }

    public function testCallEndpointAsync()
    {
        # can send multiple calls at the same time

        $route1 = [
            'endpoint' => '/',
            'method' => 'get'
        ];
        $payload1 = new RequestBag();

        $route2 = [
            'endpoint' => '/post',
            'method' => 'post'
        ];
        $payload2 = new RequestBag();

        $routes = [
            'req1' => [$route1, $payload1],
            'req2' => [$route2, $payload2]
        ];

        $returnData1 = [
            'req1' => true
        ];

        $returnData2 = [
            'req2' => true
        ];

        $response1 = \Mockery::mock(ResponseInterface::class);
        $response1->shouldReceive('getBody')
            ->andReturn(json_encode($returnData1));

        $promise1 = new FulfilledPromise($response1);

        $response2 = \Mockery::mock(ResponseInterface::class);
        $response2->shouldReceive('getBody')
            ->andReturn(json_encode($returnData2));

        $promise2 = new FulfilledPromise($response2);

        $guzzle = \Mockery::mock(Client::class)->makePartial();

        $guzzle->shouldReceive('sendAsync')
            ->andReturns($promise1, $promise2);

        $authHeader = "Bearer sometoken";
        $url = "http://someurl.com";

        $actor = \Mockery::mock(CanMakeCall::class);
        $actor
            ->shouldReceive('getAuthorizationHeader')
            ->andReturn($authHeader);
        $actor->shouldReceive('getApiUrl')
            ->andReturn($url);

        $caller = \Mockery::mock(Caller::class, [$actor])
            ->makePartial()
            ->shouldAllowMockingProtectedMethods();

        $caller->shouldReceive('getGuzzleClient')
            ->andReturn($guzzle);

        $res = $caller->callEndpointAsync($routes);

        foreach ($routes as $key => $route) {
            $this->assertArrayHasKey($key, $res);
        }

        $req1Res = $res['req1'];
        $req2Res = $res['req2'];

        $this->assertEqualsCanonicalizing($returnData1, (array) $req1Res);
        $this->assertEqualsCanonicalizing($returnData2, (array) $req2Res);
    }

}
