# Usage

## The `Api` object
The recommended entry point to the SDK is the `Api` object. A valid `Api` object can be constructed as shown below:

```php
require 'vendor/autoload.php';

use Olympus\Api;

$apiBaseUrl = "<url to olympus api>";
$oauthClientId = "<client id>";
$oauthClientSecret = "<oauth client secret>";

$api = new Api($apiBaseUrl, $oauthClientId, $oauthClientSecret);
```

## Making Requests and Using Actors

There are two authentication schemes available on the Olympus API and these determine how requests are to be made and
which entity is able to make specific requests.

### Basic Authentication
Olympus API uses Basic Authentication for:

- API health check request
- User registration requests,
- Obtaining oauth tokens for a valid user or client,
- Refreshing or revoking an access token.

All other requests require a valid access token.

To make requests requiring Basic authentication, the `Api` object can be used.
```php
# make a health check on the API
$healthCheck = $api->server->healthCheck();

# register a user
$body = [
    'reg_type' => \Olympus\Api::PASSWORD_AUTH_TYPE,
    'reg_params' => [
        'email' => 'hello@mobnia.com',
        'password' => 'nothingtoosimple4sure'
    ]   
];
$payload = new \Olympus\Helpers\RequestBag([], [], $body);

# call is made using basic authentication
$registeredUser = $api->user->register($payload);
```

### Oauth2 Authentication

To make calls which require a valid oauth2 access token, you must use an `Actor`. There are two kinds of actors:

- Client
- User

The `Client` actor makes authenticated requests using a client access token while the `User` actor makes requests
using a user access token.

They can be obtained from the api object as shown below:
```php

$clientActor = $api->getClientActor();

$userAuthType = \Olympus\Api::PASSWORD_AUTH_TYPE;
$userAuthParams = [
    'email' =>  'hello@mobnia.com',
    'password' => 'nothingtoosimple4sure'
];

$userActor = $api->getUserActor($userAuthType, $userAuthParams);
```

The actor objects can then be used to make requests:
```php

# get a user's profile
$payload = new \Olympus\Helpers\RequestBag(["id" => 2], [], []);
$profile = $clientActor->user->getProfile($payload);

# list a user's organisation
$payload = new \Olympus\Helpers\RequestBag(["id" => 2], [], []);
$userOrganisations = $userActor->listUserOrganisations($payload);
```

As a convenience, it is assumed that when making requests which require a `user_id` either in the path or in the body, the
`user_id` belonging to the authenticated `$userActor` object should be used by default. As a consequence the last example
above could be written as:
```php

# the sdk knows to automatically inject the $userActor's user id into the request
$userOrganisations = $userActor->listUserOrganisations();
```

Of course if this is not the intended behaviour, the `id` parameter can be explicitly passed via a `RequestBag` object

**Notes**

Some items are worth noting:

-  All requests are available to all actors and the api object. It is then up to the user of the sdk to use the proper
actor when making a request. For example the following requests would fail:
```php

# would fail because it requires at least a valid access token
$payload = new \Olympus\Helpers\RequestBag(["id" => 1], [], []);
$profile = $api->user->getProfile($payload);

# Would also fail because this call must be made by the user who owns the profile
$payload = new \Olympus\Helpers\RequestBag(["id" => 1], [], ["firstname" => "Wrong"]);
$updatedProfile = $clientActor->user->updateUserProfile($payload);
``` 

Please check the API documentation to check which actor is best suited for each endpoint.

### Making Requests

To make a request via the SDK, a `RequestBag` should be passed to the endpoint being called. An example is shown below:
```php
# update an organisation 
# assumed that the `$userActor` object has permission to update the organisation

# parameters that would be in the uri of the api endpoint
$path = [
    'id' => $organisation->id
];
# parameters that should be passed as query parameters (in this case it is empty)
$query = [];

# parameters that should be passed in the body of the request
$body = [
    'name' => 'new org',
    'description' => 'updated via the api'
];
$bag = new \Olympus\Helpers\RequestBag($path, $query, $body);

$updatedOrganisation = $userActor->organisation->update($bag);
```


### Manually Creating an Actor

It might be the case that an actor should be manually constructed. A very good example of this is in the case of
the registered user.

The registration call would return both a user object and a valid token details. The example below shows how to construct
a valid `$userActor`:
```php
$body = [
    'reg_type' => \Olympus\Api::PASSWORD_AUTH_TYPE,
    'reg_params' => [
        'email' => 'hello@mobnia.com',
        'password' => 'nothingtoosimple4sure'
    ]   
];

# call is made using basic authentication
$payload = new \Olympus\Helpers\RequestBag([], [], $body);
$registeredUser = $api->user->register($payload);

$accessToken = $registeredUser->data->token_data->access_token;
$refreshToken = $registeredUser->data->token_data->refresh_token;
$expiresIn = $registeredUser->data->token_data->expires_in;
$scopes = "read write";
$expiresAt = time() + $expiresIn;
$userData = $registeredUser->data->user;
$uid = $userData->id;

$userActor = new \Olympus\Actors\User($api, $accessToken, $expiresAt, $uid, $refreshToken, $scopes);
$userActor->setUserAuthData($userData);
``` 

### Refreshing the Access Token

For actors which support a refresh token as is the case for the `$userActor`, refreshing the access token is automatically
handled. The actor keeps track of the expiry time for the access token and internally makes a call to the API to refresh
token api endpoint to refresh the token.

### Responses from the API
All responses from the api are `\StdClass` objects.

### Handling Exceptions
All failed requests to the API throw an `\Olympus\Exception\OlympusException` error.
```php
try {
    # this would fail because the authentication would not be valid
    $payload = new \Olympus\Helpers\RequestBag(["id" => 1], [], []);
    $profile = $api->user->getProfile($payload);
} catch (\Olympus\Exception\OlympusException $e) {
    $httpStatusCode = $e->getHttpStatusCode();
    $apiErrorCode = $e->getErrorCode();
    $apiErrorMessage = $e->getMessage();
}
```
`\Olympus\Exception\OlympusException` errors can also be thrown in the case of improper arguments passed when making a request.
In such a case there would be no valid `$apiErrorCode` and the `$httpStatusCode` would be set to a 400.

### Making Manual Requests
While the sdk attempts to mirror all the endpoints available on the API, it is possible that development on the API
goes faster than the SDK. In such a case it is possible to manually make a request:
```php
$newEndpoint = 'users/{id}/new/not-developed/needs-auth';
$route = [
    \Olympus\Routes\RouteInterface::ENDPOINT_KEY => $newEndpoint,
    \Olympus\Routes\RouteInterface::METHOD_KEY => \Olympus\Routes\RouteInterface::POST_METHOD,
    \Olympus\Routes\RouteInterface::QUERY_PARAMS_KEY => [
        'start', 'finish'
    ],
    \Olympus\Routes\RouteInterface::PARAMS_KEY => [
            'age', 'dob'
    ],
    \Olympus\Routes\RouteInterface::PATH_KEY => [
            'id'
    ],
    \Olympus\Routes\RouteInterface::REQUIRES_AUTH => true
];

$path = [
    "id" => 1
];
$query = [
    'start' => 100,
    'finish' => 200,
];
$body = [
    'age' => 20,
    'dob' => "2020-04-18"
];
$payload = new \Olympus\Helpers\RequestBag($path, $query, $body);

$result = $userActor->getCaller()->callEndpoint($route, $payload);
```