<?php


namespace Olympus\Http;

use GuzzleHttp\Psr7\Request;
use Olympus\Actors\ActorInterface;
use Olympus\Contracts\CanMakeCall;
use Olympus\Exception\OlympusException;
use Olympus\Helpers\RequestBag;
use Olympus\Routes\RouteInterface;
use Psr\Http\Message\RequestInterface;

class RequestBuilder
{
    /**
     * @var CanMakeCall
     */
    protected $actor;

    /**
     * @var array
     */
    protected $route;

    /**
     * @var array
     */
    protected $payload = [];

    /**
     * @var array
     */
    protected $queryParams = [];

    /**
     * @var array
     */
    protected $pathParams = [];

    /**
     * @var array
     */
    protected $bodyParams = [];

    /**
     * @var string
     */
    protected $endpoint;

    /**
     * @var RequestInterface
     */
    protected $request;

    /**
     * RequestBuilder constructor.
     * @param CanMakeCall $actor
     * @param array $route
     * @param RequestBag $payload
     */
    public function __construct(CanMakeCall $actor, array $route, RequestBag $payload)
    {
        $this->actor = $actor;
        $this->route = $route;
        $this->payload = $payload;

        $this->injectUidMap();
    }

    public function injectUidMap()
    {
        if (!isset($this->route[RouteInterface::ACTOR_UID_MAP])) {
            return;
        }

        if (!($uid = $this->actor->getUid())) {
            return;
        }

        $uidMap = $this->route[RouteInterface::ACTOR_UID_MAP];
        foreach ($uidMap as $type => $value) {
            $params = $this->payload->getParams($type);
            if (isset($params[$value])) {
                continue;
            }
            $this->payload->updateParams($type, [
                $value => $uid
            ]);
        }
    }

    /**
     * @throws OlympusException
     */
    protected function extractQueryParams()
    {
        $this->queryParams = $this->extractParams(RouteInterface::QUERY_PARAMS_KEY);
        return $this;
    }

    /**
     * @param string $type
     * @param array $bag
     * @param array $required
     * @return bool
     * @throws OlympusException
     */
    protected function hasAllRequired(string $type, array $bag, array $required): bool
    {
        foreach ($required as $item) {
            if (!in_array($item, $bag)) {
                throw new OlympusException("Missing required ${type} parameter: ${item}");
            }
        }

        return true;
    }

    /**
     * @param $type
     * @return array|mixed
     * @throws OlympusException
     */
    protected function extractParams($type)
    {
        $params = $this->payload->getParams($type)->jsonSerialize();

        $requiredParams = $this->route[RouteInterface::REQUIRED_KEY] ?? [];
        $requiredParamsForType = $requiredParams[$type] ?? [];

        $this->hasAllRequired($type, array_keys($params), $requiredParamsForType);

        return $params;
    }

    /**
     * @throws OlympusException
     */
    protected function extractPathParams()
    {
        $endpointUri = $this->route[RouteInterface::ENDPOINT_KEY] ?? null;
        if (!$endpointUri) {
            throw new OlympusException("Route missing endpoint");
        }

        $this->pathParams = $this->extractParams(RouteInterface::PATH_KEY);

        $this->endpoint = $this->putPathArgsInEndpoint($endpointUri);

        return $this;
    }

    protected function putPathArgsInEndpoint($endpoint)
    {
        foreach ($this->pathParams as $key => $value) {
            $endpoint = str_replace('{' . $key . '}', $value, $endpoint);
        }

        return $endpoint;
    }

    /**
     * @throws OlympusException
     */
    protected function extractBodyParams()
    {
        $this->bodyParams = $this->extractParams(RouteInterface::PARAMS_KEY);
        return $this;
    }

    /**
     * @throws OlympusException
     */
    public function build()
    {
        $this->extractPathParams();
        $this->extractQueryParams();
        $this->extractBodyParams();

        $url = $this->actor->getApiUrl() . $this->endpoint;

        $method = $this->route[RouteInterface::METHOD_KEY] ?? RouteInterface::GET_METHOD;

        $headers = [
            'User-Agent' => 'prowork/olympus v1.0.0'
        ];

        $hasAuth = $this->route[RouteInterface::REQUIRES_AUTH] ?? true;
        if ($hasAuth) {
            $authHeader = $this->actor->getAuthorizationHeader();
            if (!$authHeader) {
                throw new OlympusException("The actor object has not been authenticated");
            }
            $headers['Authorization'] = $authHeader;
        }

        if (!empty($this->queryParams)) {
            $url .= "?".http_build_query($this->queryParams, null, '&', PHP_QUERY_RFC3986);
        }

        $body = null;
        if (!empty($this->bodyParams)) {
            $body = json_encode($this->bodyParams);
            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new OlympusException("Unable to encode body for request");
            }

            $headers['Content-Type'] = "application/json";
        }


        $this->request = new Request($method, $url, $headers, $body);

        return $this;
    }

    /**
     * @return RequestInterface
     */
    public function getRequest()
    {
        return $this->request;
    }

    /**
     * @return array|RequestBag
     */
    public function getPayload()
    {
        return $this->payload;
    }

    /**
     * @return array
     */
    public function getBodyParams()
    {
        return $this->bodyParams;
    }

    /**
     * @return string
     */
    public function getEndpoint()
    {
        return $this->endpoint;
    }

    /**
     * @return array
     */
    public function getQueryParams()
    {
        return $this->queryParams;
    }

    /**
     * @return array
     */
    public function getPathParams()
    {
        return $this->pathParams;
    }
}
