<?php


namespace Olympus\Http;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\Exception\RequestException;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\MessageFormatter;
use GuzzleHttp\Middleware;
use Monolog\Handler\StreamHandler;
use Monolog\Logger;
use Olympus\Contracts\CanMakeCall;
use Olympus\Exception\OlympusException;
use Olympus\Helpers\RequestBag;

class Caller
{
    /**
     * @var CanMakeCall
     */
    protected $actor;

    /**
     * @var string
     */
    protected $logfilePath;

    protected $logFormat;

    const DEFAULT_LOG_FORMAT = '{method} {uri} || {req_body} - {res_body}';

    public function __construct(CanMakeCall $actor, $logfilePath = null, $logFormat = null)
    {
        $this->actor = $actor;

        $this->logfilePath = $logfilePath;

        $this->logFormat = $logFormat ?? self::DEFAULT_LOG_FORMAT;
    }

    /**
     * @return CanMakeCall
     */
    public function getActor()
    {
        return $this->actor;
    }

    /**
     * @return Client
     * @throws \Exception
     */
    protected function getGuzzleClient()
    {
        $logger = null;
        if ($this->logfilePath) {
            $logHandler = new StreamHandler($this->logfilePath);
            $logger =  new Logger('Prowork-Olympus-Logger');
            $logger->pushHandler($logHandler);
        }

        $opts = [];

        if ($logger) {
            $stack = HandlerStack::create();
            $stack->push(
                Middleware::log(
                    $logger,
                    new MessageFormatter($this->logFormat)
                )
            );

            $opts = array_merge($opts, [
                'handler' => $stack
            ]);
        }

        return new Client($opts);
    }

    /**
     * @param $route
     * @param array $payload
     * @return mixed
     * @throws OlympusException
     * @throws GuzzleException
     */
    public function callEndPoint($route, RequestBag $payload = null)
    {
        $payload = $payload ?? new RequestBag();
        try {
            $client = $this->getGuzzleClient();
            $builder = (new RequestBuilder($this->getActor(), $route, $payload))->build();
            $response = $client->send($builder->getRequest());
            $body = json_decode((string) $response->getBody());
            if (json_last_error() != JSON_ERROR_NONE) {
                throw new OlympusException("Unable to fetch response");
            }
            return $body;
        } catch (RequestException $e) {
            if ($e->hasResponse()) {
                throw OlympusException::fromResponse($e->getResponse());
            }
            throw new OlympusException($e->getMessage());
        } catch (\Exception $e) {
            if ($e instanceof OlympusException) {
                throw $e;
            }

            throw new OlympusException($e->getMessage());
        }
    }
}
