<?php


namespace Olympus\Actors;


use Olympus\Api;
use Olympus\Exception\OlympusException;
use Olympus\Http\Caller;
use Olympus\Http\Router;


abstract class AbstractActor implements ActorInterface
{
    /**
     * @var Caller
     */
    protected $caller;

    /**
     * @var int
     */
    protected $expiresAt;

    /**
     * @var string
     */
    protected $accessToken;

    /**
     * @var string
     */
    protected $refreshToken;

    /**
     * @var Api
     */
    protected $api;

    /**
     * @var string
     */
    protected $uid;

    protected $scopes;

    public function __construct(
        Api $api,
        $accessToken = null,
        $expiresAt = null,
        $uid = null,
        $refreshToken = null,
        $scopes = null
    )
    {
        $this->api = $api;
        $this->accessToken = $accessToken;
        $this->expiresAt = $expiresAt;
        $this->uid = $uid;
        $this->refreshToken = $refreshToken;
        $this->scopes = $scopes;
    }

    public function getExpiresAt(): ?int
    {
        return $this->expiresAt;
    }

    public function getAccessToken(): ?string
    {
        return $this->accessToken;
    }

    public function getRefreshToken(): ?string
    {
        return $this->refreshToken;
    }

    public function getUid(): ?string
    {
        return $this->uid;
    }

    public function getApi(): Api
    {
        return $this->api;
    }

    public function setApi(Api $api)
    {
        $this->api = $api;
        return $this;
    }

    /**
     * @param array $params
     * @return ActorInterface
     */
    abstract public function authenticate(array $params);

    public function getApiUrl(): ?string
    {
        return $this->getApi()->getApiUrl();
    }

    /**
     * @param $name
     * @return Router
     * @throws OlympusException
     */
    public function __get($name)
    {
        return new Router($name, $this->getCaller());
    }

    public function getCaller()
    {
        if (!$this->caller) {
            $this->caller = new Caller($this, $this->getApi()->getLogFile(), $this->getApi()->getLogFormat());
        }

        return $this->caller;
    }

    public function setCaller(Caller $caller)
    {
        $this->caller = $caller;
        return $this;
    }

    public function getAuthorizationHeader(): ?string
    {
        $token = $this->getAccessToken();
        if (!$token) {
            return null;
        }
        return 'Bearer '.$token;
    }
}
